/*********************************************************************
Based on the Arduino VGA library by https://simple-circuit.com/
The Arduino VGA library is also based on the VGAX Library at https://github.com/smaffer/vgax
*********************************************************************/
//Modified and optimised by K. Jardine for the MI2955 Composite Video & VGA board, 19th March 2021
//Updated to V2.0 Dated 11th June 2021 - Added VGA display selections via links DS1 & DS2 and updated comments
//VGA display optimisations courtesy of Askild Eide
//This code is specifically for VGA for PCB Versions 2.3 and above

#include <avr/pgmspace.h>
#if defined(ARDUINO) && ARDUINO >= 100
  #include "Arduino.h"
#else
  #include "WProgram.h"
#endif

#include <stdlib.h>
#include "VGA_PCBV2_V2.0.h"

static bool flip = true; //Flag used to alternate the line repetition
uint8_t stretch = 0;  //Stretch display by adding extra lines
bool process_image = false; //Flag that enables image processing when rows are valid

void VGA::begin() {
  //Set up the Output pins
  pinMode(VOE, OUTPUT); //Port D6, Video Out Enable as an Output
  PORTD &= ~_BV(VOE); //Port D6, Set the Video Out En pin low.
  pinMode(VCC, OUTPUT); //Port D11, Vertical Counter Clear as an Output
  pinMode(HCC, OUTPUT); //Port D10, Horizontal Counter Clear as an Output    
  pinMode(VCClk, OUTPUT); //Port D7, Set Vertical Counter Clock as an Output
  pinMode(UserJ8pin1, OUTPUT); //Port D5, VGA Vertical Sync Output 
  pinMode(UserJ8pin6, OUTPUT); //Port D3, VGA Horizontal Sync Output
  pinMode(FS_LS, OUTPUT); //Port D9, Composite Video Frame Sync/Line Sync Output
  PORTB &= ~_BV(FS_LSPortBpin1); //Port D9, PortB pin 1, Disable the Composite Video Frame Sync/Line Sync drive by grounding the output

  // Disable TIMER0 interrupt
  TIMSK0 = 0;
  TCCR0A = 0;
  TCCR0B = 0;
  OCR0A  = 0;
  OCR0B  = 0;
  TCNT0  = 0;

  //Disable TIMER1 interrupt
  TIMSK1 = 0;  
  TCCR1A = 0;
  TCCR1B = 0;
  OCR1A = 0;
  OCR1B = 0;
  TCNT1 = 0;  

  //TIMER2 - Generates horizontal sync pulses
  TCCR2A = bit(WGM20) | bit(WGM21) | bit(COM2B1); //Pin3=COM2B1
  TCCR2B = bit(WGM22) | bit(CS21); //Divide by 8 prescaler giving a 0.5uS clock pulse for the timer.
  //Note that the values of 2 used below for HSYNCPeriod & HSYNCPulse derive from 1/(Timer2 clock pulse) ie 1/0.5 = 2
  OCR2A  = HSYNCPeriod; // 28.44uS x 2 = 56.88 (minus one) = 56 for HSYNC to pin D3 for 28.44uS period (35.15625 kHz)
  OCR2B  = HSYNCPulse; // 2uS x 2 = 4 (minus one) = 3 for HSYNC pulse width of 2uS duration. 
  TIFR2  = bit(OCF2B);  // Clear Compare Match B flag
  TIMSK2 = bit(OCIE2B); // Enable Compare Match B interrupt

  //Enable Pin Change on Interrupts
  PCMSK0 = 0; //Disable all interrupts
  PCMSK1 = bit(PCINT10)|bit(PCINT11); //Enable Interrupts for Ports A2 & A3 - Display Select links DS1 & DS2
  PCMSK2 = 0; //Disable all interrupts
  PCMSK0 = bit(PCINT0); //Enable Interrupts for Port D8 - User J8 pin 3 - CV/VGA select
  PCIFR = 0; //Clear all interrupt flags
  PCICR = bit (PCIE1)| bit(PCIE0); //Enable all Pin Change on Interrupts
  EICRA = 0;//Clear external interrupts for INT0 & INT1
  EIMSK = 0;//Disable external interrupts for INT0 & INT1    
  sei(); //Enable interrupts
}

// Horizontal Sync interrupt
ISR(TIMER2_COMPB_vect) { // Timer2 COMPB Interrupt Service Routine
   if (process_image == true){ //Perform image processing when rows are valid
    //Controls the image placement across the line and down the lines
    //Every line can be repeated so that the 256 x MI2955 lines become 512 lines or more
    //The two code paths for each line should have equal timing so that no image distortion occurs.
    //The delay values have been optimised to minimise flicker
    asm volatile (
      //This enables the start point of the image across the line
      "delay2 \n\t" // This provides a fine tuning start point delay      
      "svprts76 %[port]\n\t" //Save PORTD to R16 and set the Vertical Counter Clock & Video Output Enable bits in R16
      "out %[port], r16 \n\t" //Write R16 to port D
      //Test for a line to repeat
      "cpi %[FLIP],1\n\t"
      "brne loop2\n\t"
      "delay1\n\t"
    "loop1:\n\t"
      //Waits until we reach the end of the image across the line
      "dec %[WIDE]\n\t" 
      "delay5\n\t" // Delays x clock cycles which when looped by the width becomes a delay of (x+3) x width clock cycles
      "brne loop1\n\t"
      "delay1\n\t" // This provides a fine tuning delay ie (x+3) x width + 3 clock cycles
      //Skips a line whilst inhibiting the output display
      "svprtc6 %[port]\n\t" //Save PORTD to R16 and clear Video Output Enable bit in R16
      "out %[port], r16 \n\t" //Write R16 to port D 
      "rjmp end1\n\t" 
    "loop2:\n\t"
      //Waits until we reach the end of the image across the line   
      "dec %[WIDE]\n\t" 
      "delay5\n\t" //Delays x clock cycles which when looped by the width becomes a delay of (x+3) x width clock cycles
      "brne loop2\n\t"
      "delay1\n\t" // This provides a fine tuning delay ie (x+3) x width + 3 clock cycles
      //Clocks the Vertical Line Counter to the next line and inhibits the output display 
      "svprtc76 %[port]\n\t"  //Save PORTD and clear Vertical Counter Clock & Video Output Enable bits in R16
      "out %[port], r16 \n\t" //Write R16 to Port D 
    "end1:\n\t"
  	:
  	: [port] "I" (_SFR_IO_ADDR(PORTD)),
  	[WIDE] "r" (width),
  	[LINE] "r" (line),
  	[FLIP] "r" (flip)
  	: "r16" //clobber
  	);
	
  	//Resets the Horizontal Address Counters after each line is displayed
  	PORTB &= ~(_BV(HCCPortBpin2)); //Port D10, Set the HCC pin Low. ie Clear the Horizontal Counter to zero			
  	PORTB |= _BV(HCCPortBpin2); //Port D10, Set the HCC pin high. ie Release the Horizontal Counter to count
    
     //Implements stretching of the image when activated by flipping the lines as required
    asm volatile(
      "cpi %[INTERVAL],0 \n" //Test if interval is zero 
      "brne bypass1 \n\t" //Go if stretching is activated
      "delay3\n\t" //Synchronisation delay
      "rjmp bypass3\n\t"            
    "bypass1:\n\t" //Check for image stretching        
      "cp %[STRETCH], %[INTERVAL] \n" //Compare the stretch value with the interval value          
      "brne bypass2 \n\t" //Go if stretching is being done
      "clr %[STRETCH] \n\t" //Reset the stretch value to zero
      "rjmp end3\n\t"      
      "bypass2:\n\t" //Count the image stretching
      "inc %[STRETCH] \n\t" //Increment the stretch value by 1
    "bypass3:\n\t"            
      "com %[FLIP]\n\t" //Flips between the lines to repeat lines 
    "end3:\n\t"                                            
    : [STRETCH] "+r" (stretch), [FLIP] "+r" (flip)
    : [INTERVAL] "r" (interval)
    :
    );
   
    process_image = false; //Flag to enable image processing when rows are valid
    line++; //Increments the line counter during image processing
    if (line >= startrow) { //Only process images greater than the Start row 
      if (line <= endline) { //Only process images less than the End line
        process_image = true; //Flag to enable image processing when rows are valid
        asm volatile (  //1 cycle synchronisation delay        
          "delay1\n\t"
        );         
      }
    }
    return; //Terminates the image processing for image lines
    }
  line++; //Increments the line counter when not processing images
  process_image = false; //Flag to enable image processing when rows are valid
  if (line >= startrow) { //Only process images greater than the Start row 
    if (line <= endline) { //Only process images less than the End line
      process_image = true; //Flag to enable image processing when rows are valid
        asm volatile (  //1 cycle synchronisation delay        
          "delay1\n\t"
        );       
    }   
  }
  
  asm volatile (  //1 cycle synchronisation delay        
    "delay1\n\t"
  ); 
         
  // The Vertical Sync frame is 625 lines for a 600 line display
  if(line >= VSYNCLineStart) {
    PORTD &= ~(_BV(UserJ8pin1)); //Port D5, Set the Vertical Sync pulse low. The Vertical Sync pulse duration = 2 lines
    if(line >= VSYNCLineStop) {
      PORTD |= _BV(UserJ8pin1); //Port D5, Set the Vertical Sync pulse high
      line = 0; //Reset the line counter
      flip = true; //Reset the flag used to alternate the line repetition
      stretch = start_line_offset; //Resets the stretch count for the first line repetition. Valid values: 2,4
      //Pulse the Vertical Line counter by one character row (8 lines) to scroll the image to the top of the screen
      PORTB &= ~(_BV(VCCPortBpin3)); //Port D11, Set the VCC pin Low. ie Clear the Vertical Counter to zero   
      PORTB |= _BV(VCCPortBpin3); //Port D11, Set the VCC pin high. ie Release the Vertical Counter to count
      PORTD |= _BV(VCClk); //Port D7, Set the Vertical Counter Clock pin High.  Set up for clocking the Vertical counter
      PORTD &= ~(_BV(VCClk)); //Port D7, Set the Vertical Counter Clock pin Low which increments the Vertical counter by 1  
      PORTD |= _BV(VCClk); //Port D7, Set the Vertical Counter Clock pin High.  Set up for clocking the Vertical counter
      PORTD &= ~(_BV(VCClk)); //Port D7, Set the Vertical Counter Clock pin Low which increments the Vertical counter by 1    
      PORTD |= _BV(VCClk); //Port D7, Set the Vertical Counter Clock pin High.  Set up for clocking the Vertical counter
      PORTD &= ~(_BV(VCClk)); //Port D7, Set the Vertical Counter Clock pin Low which increments the Vertical counter by 1    
      PORTD |= _BV(VCClk); //Port D7, Set the Vertical Counter Clock pin High.  Set up for clocking the Vertical counter
      PORTD &= ~(_BV(VCClk)); //Port D7, Set the Vertical Counter Clock pin Low which increments the Vertical counter by 1  
      PORTD |= _BV(VCClk); //Port D7, Set the Vertical Counter Clock pin High.  Set up for clocking the Vertical counter
      PORTD &= ~(_BV(VCClk)); //Port D7, Set the Vertical Counter Clock pin Low which increments the Vertical counter by 1  
      PORTD |= _BV(VCClk); //Port D7, Set the Vertical Counter Clock pin High.  Set up for clocking the Vertical counter
      PORTD &= ~(_BV(VCClk)); //Port D7, Set the Vertical Counter Clock pin Low which increments the Vertical counter by 1      
      PORTD |= _BV(VCClk); //Port D7, Set the Vertical Counter Clock pin High.  Set up for clocking the Vertical counter
      PORTD &= ~(_BV(VCClk)); //Port D7, Set the Vertical Counter Clock pin Low which increments the Vertical counter by 1      
      PORTD |= _BV(VCClk); //Port D7, Set the Vertical Counter Clock pin High.  Set up for clocking the Vertical counter
      PORTD &= ~(_BV(VCClk)); //Port D7, Set the Vertical Counter Clock pin Low which increments the Vertical counter by 1 

      PORTB |= _BV(HCCPortBpin2); //Port D10, Horizontal Counter Clear Output, PortB Pin 2, output high to ensure that it is ready to count. 
    }
    return;
  }
  return;  
}
