/*********************************************************************
Based on the Arduino VGA library by https://simple-circuit.com/
The Arduino VGA library is also based on the VGAX Library at https://github.com/smaffer/vgax
*********************************************************************/
//Modified and optimised by K. Jardine for the MI2955 Composite Video & VGA board, 19th March 2021
//This code is specifically for Composite Video for PCB Version 3.1

#include <avr/pgmspace.h>
#if defined(ARDUINO) && ARDUINO >= 100
  #include "Arduino.h"
#else
  #include "WProgram.h"
#endif

#include <stdlib.h>
#include "CV_PCBV3_V1.0.h"

int loopcount; //Holder for scrolling the output
int loopcounter; //Counter for scrolling the output

uint16_t CYCLES_SCANLINE; //Holder for the Horizontal Sync period
uint16_t CYCLES_HORZ_SYNC; //Holder for the Horizontal Sync pulse duration
uint16_t FRAME_LINES; //Holder for the number of lines in a frame
uint16_t CYCLES_VERT_SYNC; //Holder for the Horizontal Sync pulse duration (Inverted Line Sync) during the Vertical Sync pulse
uint16_t LINE_CHANGE; //Holder for the line where the Inverted Line Sync reverts to Non Inverted Line Sync. Inverted Line Sync occurs during the Vertical Sync pulse
uint16_t START_LINE; //Holder for the start line
uint16_t END_LINE; //Holder for the end line
uint16_t CYCLES_OUTPUT_START; //Holder for the starting position across the line
uint8_t VIDEO_WIDTH; //Holder for the video width
int correction_offset; //Holder for the number of character rows offset for the reference display position

void CV::begin(bool NTSC_PAL) {
  //Set up for either NTSC or PAL
  if (NTSC_PAL == true){ //PAL
   loopcount = PALscrollRows; //Clocks the Vertical Counter for PAL to align the image at the top of the screen
   CYCLES_SCANLINE = PAL_CYCLES_SCANLINE; //Sets the Horizontal Sync period
   CYCLES_HORZ_SYNC = PAL_CYCLES_HORZ_SYNC; //Sets the Horizontal Sync pulse duration
   FRAME_LINES = PAL_LINES; //Sets the number of lines in a frame
   CYCLES_VERT_SYNC = PAL_CYCLES_VERT_SYNC; //Sets the Horizontal Sync pulse duration (Inverted Line Sync) during the Vertical Sync pulse
   LINE_CHANGE = PAL_LINE_CHANGE; //Sets the line where the Inverted Line Sync reverts to Non Inverted Line Sync. Inverted Line Sync occurs during the Vertical Sync pulse 
   START_LINE  = PAL_START_LINE; //Sets the start line
   END_LINE = PAL_END_LINE; //Sets the end line
   CYCLES_OUTPUT_START = PAL_CYCLES_OUTPUT_START; //Sets the starting position across the line
   VIDEO_WIDTH = PAL_VIDEO_WIDTH; //Sets the video width
   correction_offset = PALCcorrectionOffset; //The number of character rows offset when in PAL to set the reference display position
  }
  else{ //NTSC
   loopcount = NTSCscrollRows; //Clocks the Vertical Counter for NTSC to align the image at the top of the screen
   CYCLES_SCANLINE = NTSC_CYCLES_SCANLINE; //Sets the Horizontal Sync period
   CYCLES_HORZ_SYNC = NTSC_CYCLES_HORZ_SYNC; //Sets the Horizontal Sync pulse duration
   FRAME_LINES = NTSC_LINES; //Sets the number of lines in a frame
   CYCLES_VERT_SYNC = NTSC_CYCLES_VERT_SYNC; //Sets the Horizontal Sync pulse duration (Inverted Line Sync) during the Vertical Sync pulse
   LINE_CHANGE = NTSC_LINE_CHANGE; //Sets the line where the Inverted Line Sync reverts to Non Inverted Line Sync. Inverted Line Sync occurs during the Vertical Sync pulse
   START_LINE  = NTSC_START_LINE; //Sets the start line
   END_LINE = NTSC_END_LINE; //Sets the end line
   CYCLES_OUTPUT_START = NTSC_CYCLES_OUTPUT_START; //Sets the starting position across the line
   VIDEO_WIDTH = NTSC_VIDEO_WIDTH; //Sets the video width 
   correction_offset = NTSCcorrectionOffset; //The number of character rows offset when in NTSC to set the reference display position        
  }
  
  //Set up the Output pins
  pinMode(VOE, OUTPUT); //Port D6, Video Out Enable as an Output
  pinMode(PE_A2, OUTPUT); //Port D11, Vertical Line Counter Scroll bit 2 output
  pinMode(PE_A1_HCC, OUTPUT); //Port D10, Vertical Line Counter Scroll bit 1 output, Horizontal Counter Clear Output  
  pinMode(VCClk, OUTPUT); //Port D7, Set Vertical Counter Clock as an Output
  pinMode(FS_LS, OUTPUT); //Port D9, Composite Video Frame Sync/Line Sync Output
  DDRC |= _BV(PE_A0_PortCpin0); //Port A0, Vertical Line Counter Scroll bit 0 output
  pinMode(PE_A3, OUTPUT); //Port D12, Vertical Line Counter Scroll bit 3 output
  pinMode(PE_A4, OUTPUT); //Port D13, Vertical Line Counter Scroll bit 4 output
  DDRC |= _BV(PE__PortCpin1); // Port A1, Vertical Line Counter Parallel Enable output
  PORTC |= _BV(PE__PortCpin1); //Port A1, Set the Vertical Line Counter Parallel Enable high
    
  //Disable TIMER0 interrupt
  TIMSK0 = 0;
  TCCR0A = 0;
  TCCR0B = 0;
  OCR0A  = 0;
  OCR0B  = 0;
  TCNT0  = 0;

  //Disable TIMER2 interrupt
  TIMSK2 = 0;
  TCCR2A = 0;
  TCCR2B = 0;
  OCR2A  = 0;
  OCR2B  = 0;
  TCNT2  = 0;

  //Inverted fast pwm mode on timer 1
  TCCR1A = _BV(COM1A1) | _BV(COM1A0) | _BV(WGM11);
  TCCR1B = _BV(WGM13) | _BV(WGM12) | _BV(CS10);
 	ICR1 = CYCLES_SCANLINE; //Sets the Horizontal Sync period
  OCR1A = CYCLES_HORZ_SYNC; //Sets the Horizontal Sync pulse duration
  TIMSK1 = _BV(TOIE1); //Enable Interrupt

  //Enable Pin Change on Interrupts
  PCMSK0 = 0; //Disable all interrupts
  PCMSK1 = 0; //Disable all interrupts
  PCMSK2 = 0; //Disable all interrupts
  PCMSK0 = bit(PCINT0); //Enable Interrupts for Port D8 - User J8 pin 3 - CV/VGA select
  PCMSK1 = bit(PCINT10)|bit(PCINT11); //Enable Interrupts for Ports A2 & A3 - Display Select links DS1 & DS2
  PCIFR = 0; //Clear all interrupt flags
  PCICR = bit (PCIE1)| bit(PCIE0); //Enable all Pin Change on Interrupts
  EICRA = bit(ISC10)|bit(ISC00);//Set for any logical change for external interrupts D3 Scroll INT1, D2 PAL/NTSC INT0
  EIMSK = bit(INT1)|bit(INT0);//Enable external interrupts for D3 Scroll INT1, D2 PAL/NTSC INT0 
  sei(); //Enable interrupts
}

// Horizontal Sync interrupt
ISR(TIMER1_OVF_vect) { //Timer1 Overflow Interrupt Service Routine
  line++; //Increment the line counter
  if(line >= FRAME_LINES) { //Create the Vertical sync pulse when the end of the frame is reached
	  OCR1A = CYCLES_VERT_SYNC; //The Horizontal Sync pulse duration (Inverted Line Sync) during the Vertical Sync pulse
    line     = 0; //Reset the line counter
    PORTB &= ~(_BV(PE_A1_HCCPortBpin2)); //Port D10, Set the HCC pin Low. ie Clear the Horizontal Counter to zero     
    PORTB |= _BV(PE_A1_HCCPortBpin2); //Port D10, Set the HCC pin high. ie Release the Horizontal Counter to count
     
    __asm__ __volatile__( //Port A0, PE_A0_PortCpin0, set or clear the Vertical Line Counter Scroll bit A0
      "clr r17\n\t"
      "or r17,%[CORRECTION]\n\t" //Bit0 will be either set or cleared depending upon the Correction value
      "PE_A0 %[port]\n\t"  //Port A0, PE_A0_PortCpin0, Save the Vertical Line Counter Scroll bit A0       
    : 
    :[port] "I" (_SFR_IO_ADDR(PORTC)),
     [CORRECTION] "r" (correction_offset)   
    );
   
    __asm__ __volatile__( //Ports D10-D13, Set the Vertical Line Counter Scroll bits A1-A4 output
      "clr r17\n\t"     
      "or r17,%[CORRECTION]\n\t" //Bits 1-4 will be either set or cleared depending upon the Correction value
      "PE_A1_4 %[port]\n\t" //Ports D10-D13, Save the Vertical Line Counter Scroll bits A1-A4
    : 
    :[port] "I" (_SFR_IO_ADDR(PORTB)),
    [CORRECTION] "r" (correction_offset)   
    );

    if (bool_scroll == false){ //Scroll if false. No scroll if true.    
      __asm__ __volatile__(
      "clr r17\n\t"
      "or r17,%[CORRECTION]\n\t" //Bit0 will be either set or cleared depending upon the Correction value
      "PE_A0 %[port]\n\t" //Port A0, PE_A0_PortCpin0, Save the Vertical Line Counter Scroll bit A0            
      : 
      :[port] "I" (_SFR_IO_ADDR(PORTC)),
      [CORRECTION] "r" (correction_offset+scroll_offset+loopcount)   
      );
    
      __asm__ __volatile__( //Ports D10-D13, Set the Vertical Line Counter Scroll bits A1-A4 output
      "clr r17\n\t"    
      "or r17,%[CORRECTION]\n\t" //Bits 1-4 will be either set or cleared depending upon the Correction value
      "PE_A1_4 %[port]\n\t" //Ports D10-D13, Save the Vertical Line Counter Scroll bits A1-A4
      : 
      :[port] "I" (_SFR_IO_ADDR(PORTB)),
      [CORRECTION] "r" (correction_offset+scroll_offset+loopcount)   
      );   
    }   
    
    PORTD &= ~(_BV(VOE)); //Port D6, set the Video Out Enable pin low.
    PORTC &= ~(_BV(PE__PortCpin1)); //Port A1, Vertical Line Counter Parallel Enable output. Set the Parallel Load pin low to set up for the parallel scroll data. 
    PORTD |= _BV(VCClk); //Port D7, Vertical Counter Clock Output. Set the pin high to set up for clocking the parallel data in.
    PORTD &= ~(_BV(VCClk)); //Port D7, Vertical Counter Clock Output. Set the pin low to latch the parallel data in.   
    PORTC |= _BV(PE__PortCpin1); //Port A1, Vertical Line Counter Parallel Enable output. Set the Parallel Load pin high to disable the parallel loading.
    PORTB |= _BV(PE_A1_HCCPortBpin2); //Port D10, Vertical Line Counter Scroll bit 1, Horizontal Counter Clear Output, PortB Pin 2, output high to ensure that the Horizonal Counter is ready to count.
  
    return;
  }
  
  if (line == LINE_CHANGE) { //The line where the Inverted Line Sync reverts to Non Inverted Line Sync.
  	OCR1A = CYCLES_HORZ_SYNC; //Sets the Horizontal Sync pulse duration
    if (scroll_cntr>=0){ //Tests to see if scrolling has been requested to act as a debounce based on the frame sync rate
      scroll_cntr=scroll_cntr-1;
    }
    return;
  }
  
  if (line >= (START_LINE + start_line_offset)) { //Only process images great than the Start line
    if (line <= (END_LINE + start_line_offset + height_offset)) { //Only process images less than the End line
      
    //Determines the starting position across the line by comparing the current Timer1 value with a START value
  		__asm__ __volatile__ (
  			"subi	%[time], 10\n" //Subtracts 10 from the START value
  			"sub	%[time], %[tcnt1l]\n\t" //Subtracts the current Timer 1 value from the current START value
  		"100:\n\t"
  			"subi	%[time], 3\n\t" //Subtracts 3 from the current START value
  			"brcc	100b\n\t" //Loops until the current START value goes negative
  			"subi	%[time], 0-3\n\t" //Subtracts minus 3 which actually adds 3 to the current START value
  			"breq	101f\n\t" //Ends when the current START value = 0
  			"dec	%[time]\n\t" //Decrement by 1 as the current START value = 1 or 2
  			"breq	102f\n\t" //Ends when the current START value = 1
  			"rjmp	102f\n" //Ends when the current START value = 2
  		"101:\n\t"
  			"nop\n" 
  		"102:\n"
  		:
  		: [time] "a" (CYCLES_OUTPUT_START),
  		[tcnt1l] "a" (TCNT1L)
  	);
  	
    width = VIDEO_WIDTH; //Sets the video width   
      //Controls the image placement across the line and down the lines
      //The code paths have equal timing so that no image distortion occurs
  	    asm volatile (
  			"svprts76 %[port]\n\t" //Save PORTD to R16 and set Vertical Counter Clock & Video Output Enable bits in R16
  			"out %[port], r16 \n\t" //Write R16 to port D
  			"nop\n\t"
  		"loop3:\n\t"
        //Waits until we reach the end of the image across the line			
  			"dec %[WIDE]\n\t"	
  			"delay10\n\t" //Delays 10 clock cycles which when looped by the width becomes a delay of (10+3) x width clock cycles
  			"brne loop3\n\t"
        "delay3\n\t" //This provides a fine tuning delay ie (10+3) x width + 3 clock cycles	
  			"svprtc76 %[port]\n\t"	//Save PORTD and clear Vertical Counter Clock & Video Output Enable bits in R16	
  			"out %[port], r16 \n\t" //Write R16 to video	
  		"end2:\n\t"
  	:
  	: [port] "I" (_SFR_IO_ADDR(PORTD)),
  	[WIDE] "r" (width),
  	[LINE] "r" (line)
  	: "r16" //clobber
  	);

    //Resets the Horizontal Address Counters after each line is displayed
    PORTB &= ~(_BV(PE_A1_HCCPortBpin2)); //Port D10, Set the HCC pin Low. ie Clear the Horizontal Counter to zero     
    PORTB |= _BV(PE_A1_HCCPortBpin2); //Port D10, Set the HCC pin high. ie Release the Horizontal Counter to count 
    return;
    }
  }

 //This adds time per unused line to allow interrupt processing to occur for Scroll & PAL/NTSC changes without upsetting the display image
 EIMSK = bit(INT1)|bit(INT0); //Enable external interrupts for D3 Scroll INT1, D2 PAL/NTSC INT0
 sei(); //Enable nested interrupts

 asm volatile (
   "loop4:\n\t"
    //Delays approx. 16.25uS
    "dec %[Wait]\n\t" 
    "delay10\n\t" //Delays 10 clock cycles which when looped by the wait time becomes a delay of (10+3) x Wait clock cycles
    "brne loop4\n\t"
    :
    :
    [Wait] "r" (20)
    :
    );

  EIMSK = 0; //Disable interrupts for Ports D3 (User J8 pin6) & Port D2  
}
