/*MI2955 Display to Composite Video & VGA Converter
Dated 19th March 2021
VK2KVJ - Kevin Jardine
When used with the MI2955 Composite Video & VGA board this software converts the MI2955 video data streams to either Composite Video or 800x600 56Hz VGA.
This software is entirely interrupt driven.

This software supports PCB Version 3.1

Copyright 2020 Kevin Jardine

It is intended that all design information related to the MI2955 Composite Video & VGA board be made publically available for the personal use of those who have a MI2955 Radio Communications Test Set. 
The repository of that information is the files areas used by the Marconi Test Instruments User group at Marconi-Test-Instruments@groups.io.

Permission is hereby granted to any person to use this software as they see fit whilst noting that the software is provided without warranty of any kind.
The above copyright notice and this permission notice shall be included in all copies or substantial portions of this Software.
*/

 //Libraries
#include "VGA_PCBV3_V1.0.h" //Include the VGA library
#include "CV_PCBV3_V1.0.h" //Include the CV library
#include "CV_VGA_PCBV3_asm_macros_V1.0.h" //Include the macro library

//Definitions
//All port usage is defined here
const byte PAL_NTSC = 2; //Port D2, PAL/NTSC select
#define Display_Selection1 A2 //Port A2, Display Selection 1
#define Display_Selection2 A3 //Port A3, Display Selection 2
#define UserJ8pin2 4 //Port D4, User Spare
#define UserJ8pin3 8 //Port D8, Select for Composite Video = true or VGA = false
const byte UserJ8pin1 = 5; //Port D5, VGA Vertical Sync Output
const byte UserJ8pin6 = 3; //Port D3, VGA Horizontal Sync Output for VGA or scroll enable input with pullup for Composite Video
const byte PE_A1_HCC = 10; //Port D10, Vertical Line Counter Scroll bit 1 output, Horizontal Counter Clear Output
const byte PE_A1_HCCPortBpin2 = 2; //Port D10, Vertical Line Counter Scroll bit 1 output, Horizontal Counter Clear Output, PortB Pin 2
const byte PE_A2 = 11; //Port D11, Vertical Line Counter Scroll bit 2 output
const byte PE_A2_PortBpin3 = 3; //Port D11, Vertical Line Counter Scroll bit 2 output, PortB Pin 3
const byte FS_LS = 9; //Port D9, Composite Video Frame Sync/Line Sync Output
const byte VOE = 6; //Port D6, Video Out Enable
const byte VCClk = 7; //Port D7, Vertical Counter Clock Output
const byte PE_A0_PortCpin0 = 0; //Port A0, Vertical Line Counter Scroll bit 0 output
const byte PE_A3 = 12; //Port D12, Vertical Line Counter Scroll bit 3 output
const byte PE_A3_PortBpin4 = 4; //Port D12, Vertical Line Counter Scroll bit 3 output, PortB Pin 4
const byte PE_A4 = 13; //Port D13, Vertical Line Counter Scroll bit 4 output
const byte PE_A4_PortBpin5 = 5; //Port D13, Vertical Line Counter Scroll bit 4 output, PortB Pin 5
const byte PE__PortCpin1 = 1; // Port A1, Vertical Line Counter Parallel Enable output

//Common Variables
char Version[] = "Software Version 1.0";
char PCB[] = "PCB Version 3.1";
bool CV_VGA_Mode = true; //Composite Video mode = true (default), VGA mode = false
static uint16_t line = 0; //Line counter for counting the lines down the screen
uint8_t width; //The image width variable for counting image placement across the screen
int8_t height_offset = 8; //Default value. An offset to adjust the screen image height
int8_t start_line_offset = 0; //Default value. An offset to adjust the image position down the screen
int scroll_offset = 0; //Default value. An offset to adjust the scroll value

//Variables For Composite Video
byte DisplaySelect = 0; //Initialises the display selections - Default =0
bool PALorNTSCmode = true; //Variable to save the selected PAL or NTSC mode. NTSC=false, PAL=true.
bool bool_scroll = true; //Scroll if false. No scroll if true.
int const Debounce = 80; //The number of interrupt frame repeats before a scrolling request is next actioned
int scroll_cntr = Debounce; //To debounce the scroll button

VGA VGAdisplay; // Initialize the VGA library
CV CVdisplay; // Initialize the CV library
 
void setup(void) {
  Serial.begin(9600); //IDE port speed
  //Test for Composite Video or VGA selection    
  pinMode(UserJ8pin3,INPUT_PULLUP);  //Set up Port D8 with a pullup to logic 1
  if (digitalRead(UserJ8pin3) == true){ //Composite Video = true or VGA = false
    CVSetup();//Set up the board for Composite//The number of interrupt frame repeats before a scrolling request is next actioned Video
  }
  else{ //Set up the board for VGA
    CV_VGA_Mode = false; //Sets VGA mode
    VGAdisplay.begin();  //Initialize the display for VGA
    Serial.println ("VGA");
  }
  
  Serial.println(Version); //Prints the Software Version to the Serial Monitor
  Serial.println(PCB); //Prints the PCB Version to the Serial Monitor 
}
 
void loop() {
/*Do not put code in the loop unless performing testing.
Code running in the loop will have microprocesser instructions that will take from 1-4 clock cycles to execute. Apart from the Setup code that operates only once,
all the code used is interrupt driven. When an interrupt occurs the processor waits until the current instruction being executed completes its cycles
before the interrupt is serviced. This means that there would be 1-4 cycles of delay that would cause interrupt jitter. This would translate into visible
jitter on the screen that would cause image degradation.
*/
} 

ISR(INT0_vect) { //When a PAL/NTSC link change occurs
      CVSetup();//Set up the board for Composite Video
}

ISR(INT1_vect) { //Scroll button processing
  if (scroll_cntr < 0){ //After timing out, toggle the scroll setting and reset the count.  The actual scrolling is done by the CV software.    
    bool_scroll = !bool_scroll; //Toggle the scroll condition                
    scroll_cntr = Debounce; //Resets the debounce count which restarts the scroll_cntr whose counting is done in the Horizontal Sync interrupt routine at a 20mS/count rate
  }  
}

//Composite Video / VGA selection change interrupt
ISR(PCINT0_vect) { // PCINT0 Interrupt Service Routine for Port D8 (User J8 pin3)
  if (digitalRead(UserJ8pin3) != CV_VGA_Mode){ //Test for a CV or VGA selection change
    CV_VGA_Mode=!CV_VGA_Mode; //Flip the mode over if there was a change
    if(CV_VGA_Mode==false){ //Test the new selection for CV or VGA mode
     VGAdisplay.begin();  //Set up the board for VGA
     Serial.println ("VGA");
    }
    else{
     CVSetup();//Set up the board for Composite Video
    }
  }
}

//Composite Video Display Select change interrupts
ISR(PCINT1_vect) { //PCINT1 Interrupt Service Routine for Ports A2 & A3 (DS1 & DS2 links)
  if (digitalRead (Display_Selection1)== false){ //Change the settings if the link is changed
     DisplaySelect = DisplaySelect|1; //Set bit 0 of the DisplaySelect variable
  }
  else {
     DisplaySelect = DisplaySelect&254; //Clear bit 0 of the DisplaySelect variable
  }

  if (digitalRead (Display_Selection2)== false){ //Change the settings if the link is changed
     DisplaySelect = DisplaySelect|2; //Set bit 1 of the DisplaySelect variable
  }
  else {
     DisplaySelect = DisplaySelect&253; //Clear bit 1 of the DisplaySelect variable  
  }

  Display_Options(); //Set the display options
}

void  Display_Options(){ //Configure all the screen display options here
    switch (DisplaySelect){
      case 1: //Display Select 1 (Port A3)
            //Puts the screen title at both the top & bottom for those screens that have trouble showing the top line      
            if (PALorNTSCmode==true){ //PAL
              height_offset = 16; //An offset to adjust the screen image height
              start_line_offset = 8; //An offset to adjust the image position down the screen
              scroll_offset = 0; //An offset to adjust the scroll value
              bool_scroll = true; //Force no scrolling
            }
            //Puts the screen title at the top but expands the image down the screen to its maximum            
            else{ //NTSC - same as default
              height_offset = 16; //An offset to adjust the screen image height
              start_line_offset = 0; //An offset to adjust the image position down the screen
              scroll_offset = 0;
              bool_scroll = true; //Force No scrolling                 
            }
      break;
      case 2: //Display Select 2 (Port A2)
             //Puts the screen title at the top but moves the image further down the screen for those screens that have trouble showing the top line    
             if (PALorNTSCmode==true){ //Default display settings for PAL
              height_offset = 0; //An offset to adjust the screen image height
              start_line_offset = 16; //An offset to adjust the image position down the screen
              scroll_offset = 0; //An offset to adjust the scroll value
              bool_scroll = true; //Force No scrolling
            }
            //Puts the screen title at the top but moves the image further down the screen for those screens that have trouble showing the top line    
            else{ //NTSC
              height_offset = 8; //An offset to adjust the screen image height
              start_line_offset = 8; //An offset to adjust the image position down the screen
              scroll_offset = 0;
              bool_scroll = true; //Force No scrolling    
            }           
      break;
      case 3: //User settings for when both links DS1 and DS2 are fitted
            //Add your User code here
      break;
      default:
            if (PALorNTSCmode==true){ //Default display settings for PAL
              height_offset = 0; //An offset to adjust the screen image height
              start_line_offset = 8; //An offset to adjust the image position down the screen
              scroll_offset = 0;
              bool_scroll = true; //Force No scrolling
            }
            else{ //Default display settings for NTSC
              height_offset = 0; //An offset to adjust the screen image height
              start_line_offset = 0; //An offset to adjust the image position down the screen
              scroll_offset = 0;
              bool_scroll = true; //Force No scrolling  
            }
      break;
    }    
}

void CVSetup(){ //Set up for Composite Video
    Serial.println ("Composite Video");
    CV_VGA_Mode = true; //Sets Composite Video mode   
    pinMode(PAL_NTSC, INPUT); //Set up the PAL/NTSC select port
    pinMode(Display_Selection1, INPUT); //Set up the Display Select 1 port
    pinMode(Display_Selection2, INPUT); //Set up the Display Select 2 port
    pinMode(UserJ8pin6, INPUT_PULLUP); //Define "User" connector J8 pin 6 for a scroll enable input with pullup
    
    if (digitalRead (Display_Selection1)==false){ //Read in the display select 1 setting
      DisplaySelect = DisplaySelect|1; //Set bit 0 of the DisplaySelect variable
    }
    if (digitalRead (Display_Selection2)==false){ //Read in the display select 2 setting
      DisplaySelect = DisplaySelect|2; //Set bit 1 of the DisplaySelect variable
    }  
        
    if (digitalRead (PAL_NTSC)==false){
      PALorNTSCmode = false; //PALorNTSCmode = NTSC
      Serial.println ("NTSC");
    }
    else{
      Serial.println ("PAL");
      PALorNTSCmode = true; //PALorNTSCmode = PAL
    }

    Display_Options(); //Update the displayed selections
    CVdisplay.begin(PALorNTSCmode);  //Initialize the display
}

// End of code.
