/*
 Copyright (c) 2010 Myles Metzer

 Permission is hereby granted, free of charge, to any person
 obtaining a copy of this software and associated documentation
 files (the "Software"), to deal in the Software without
 restriction, including without limitation the rights to use,
 copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following
 conditions:

 The above copyright notice and this permission notice shall be
 included in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 OTHER DEALINGS IN THE SOFTWARE.
*/

#include <avr/interrupt.h>
#include <avr/io.h>

#include "video_gen_Mod.h" //Modified by KJ 19/8/20
#include "spec/video_properties.h"
#include "spec/asm_macros.h"
#include "spec/hardware_setup.h"

//#define REMOVE6C
//#define REMOVE5C
//#define REMOVE4C
//#define REMOVE3C

int renderLine;
TVout_vid display;
void (*render_line)();			//remove me
void (*line_handler)();			//remove me
void (*hbi_hook)() = &empty;
void (*vbi_hook)() = &empty;

// sound properties
volatile long remainingToneVsyncs;

void empty() {}

void render_setup(uint8_t mode, uint8_t x, uint8_t y, uint8_t *scrnptr) {

	display.screen = scrnptr;
	display.hres = x;
	display.vres = y;
	display.frames = 0;
	
	if (mode){
		display.vscale_const = _PAL_LINE_DISPLAY/display.vres - 1;
		display.scroll_loop=Number_PAL_Scroll_Lines; //Added by KJ 30/11/20 Sets the number of character rows to scroll in PAL.
	}
	else{
		display.vscale_const = _NTSC_LINE_DISPLAY/display.vres - 1;
		display.scroll_loop=Number_NTSC_Scroll_Lines; //Added by KJ 30/11/20 Sets the number of character rows to scroll in NTSC.		
	}
	display.vscale = display.vscale_const;
	
	//selects the widest render method that fits in 46us
	//as of 9/16/10 rendermode 3 will not work for resolutions lower than
	//192(display.hres lower than 24)
	unsigned char rmethod = (_TIME_ACTIVE*_CYCLES_PER_US)/(display.hres*8);
	switch(rmethod) {
		case 6:
			render_line = &render_line6c;
			break;
		case 5:
			render_line = &render_line5c;
			break;
		case 4:
			render_line = &render_line4c;
			break;
		case 3:
			render_line = &render_line3c;
			break;
		default:
			if (rmethod > 6)
				render_line = &render_line6c;
			else
				render_line = &render_line3c;
	}
	

	DDR_VID |= _BV(VID_PIN);
	DDR_SYNC |= _BV(SYNC_PIN);
	DDR_FSYNC_VOE |= _BV(FSYNC_VOE_PIN); //Set the Frame Sync Pin or Video Out En, as an output. Added by KJ 22/11/20
	PORT_VID &= ~_BV(VID_PIN);
	PORT_SYNC |= _BV(SYNC_PIN);
	PORT_FSYNC_VOE &= ~_BV(FSYNC_VOE_PIN); //Set the Frame Sync or Video Out En, pin low. Added by KJ 22/11/20
	DDR_SND |= _BV(SND_PIN);	// for tone generation.
	
	#if !defined(PCB_Version_2) //Added by KJ 22/11/20
		DDR_PE_A0 |= _BV(PE_A0_PIN); //Set Scroll address A0 as an output. Added by KJ 22/11/20
		DDR_PE_A1 |= _BV(PE_A1_PIN); //Set Scroll address A1 as an output. Added by KJ 22/11/20
		DDR_PE_A2 |= _BV(PE_A2_PIN); //Set Scroll address A2 as an output. Added by KJ 22/11/20
		DDR_PE_A3 |= _BV(PE_A3_PIN); //Set Scroll address A3 as an output. Added by KJ 22/11/20
		DDR_PE_A4 |= _BV(PE_A4_PIN); //Set Scroll address A4 as an output. Added by KJ 22/11/20	
		DDR_PE_ |= _BV(PE_PIN); //Set the Parallel Enable an output. Added by KJ 22/11/20	
		
		PORTD |= _BV(FSYNC_VOE_PIN);  //Set the Frame SYNC or Video Out En, pin high to enable Video output. Added by KJ 22/11/20	
		PORT_PE_ |= _BV(PE_PIN); //Set the Parallel Load pin high to disable the parallel loading. Added by KJ 22/11/20		
		//Clear the scroll address
		PORT_PE_A0 &= ~(_BV(PE_A0_PIN)); //Set the A0 scroll data = 0. Added by KJ 22/11/20
		PORT_PE_A1 &= ~(_BV(PE_A1_PIN)); //Set the A1 scroll data = 0. Added by KJ 22/11/20
		PORT_PE_A2 &= ~(_BV(PE_A2_PIN)); //Set the A2 scroll data = 0. Added by KJ 22/11/20
		PORT_PE_A3 &= ~(_BV(PE_A3_PIN)); //Set the A3 scroll data = 0. Added by KJ 22/11/20
		PORT_PE_A4 &= ~(_BV(PE_A4_PIN)); //Set the A4 scroll data = 0. Added by KJ 22/11/20		
		PORT_PE_ &= ~(_BV(PE_PIN)); //Set the Parallel Load pin low. Set up for the parallel scroll data. Added by KJ 22/11/20
		PORT_VID |= _BV(VID_PIN); //Set the Video Enable pin High.  Set up for clocking the parallel data in. Added by KJ 22/11/20
		PORT_VID &= ~(_BV(VID_PIN)); //Set the Video Enable pin Low. Latch the parallel data in.  Added by KJ 22/11/20		
		PORT_PE_ |= _BV(PE_PIN); //Set the Parallel Load pin high to disable the parallel loading. Added by KJ 22/11/20				
	#endif //Added by KJ 22/11/20
	
	// inverted fast pwm mode on timer 1
	TCCR1A = _BV(COM1A1) | _BV(COM1A0) | _BV(WGM11);
	TCCR1B = _BV(WGM13) | _BV(WGM12) | _BV(CS10);
	
	if (mode) {
		display.start_render = _PAL_LINE_MID - ((display.vres * (display.vscale_const+1))/2);
		display.output_delay = _PAL_CYCLES_OUTPUT_START;
		display.vsync_end = _PAL_LINE_STOP_VSYNC;
		display.lines_frame = _PAL_LINE_FRAME;
		ICR1 = _PAL_CYCLES_SCANLINE;
		OCR1A = _CYCLES_HORZ_SYNC;
		}
	else {
		display.start_render = _NTSC_LINE_MID - ((display.vres * (display.vscale_const+1))/2) + 8;
		display.output_delay = _NTSC_CYCLES_OUTPUT_START;
		display.vsync_end = _NTSC_LINE_STOP_VSYNC;
		display.lines_frame = _NTSC_LINE_FRAME;
		ICR1 = _NTSC_CYCLES_SCANLINE;
		OCR1A = _CYCLES_HORZ_SYNC;
	}
	display.scanLine = display.lines_frame+1;
	line_handler = &vsync_line;
	TIMSK1 = _BV(TOIE1);
	sei();
}

// render a line
ISR(TIMER1_OVF_vect) {
	hbi_hook();
	line_handler();
}

void blank_line() {
		
	if ( display.scanLine == display.start_render) {
		renderLine = 0;
		display.vscale = display.vscale_const;
		line_handler = &active_line;
	}
	else if (display.scanLine == display.lines_frame) {
		line_handler = &vsync_line;
		vbi_hook();
	}
	
	display.scanLine++;
}

void active_line() {
	wait_until(display.output_delay);
	render_line();
	if (!display.vscale) {
		display.vscale = display.vscale_const;
		renderLine += display.hres;
	}
	else
		display.vscale--;
		
	if ((display.scanLine + 1) == (int)(display.start_render + (display.vres*(display.vscale_const+1))))
		line_handler = &blank_line;
		
	display.scanLine++;
}

void vsync_line() {
	if (display.scanLine >= display.lines_frame) {
		OCR1A = _CYCLES_VIRT_SYNC;
		display.scanLine = 0;
		display.frames++;
		if (remainingToneVsyncs != 0)
		{
			if (remainingToneVsyncs > 0)
			{
				remainingToneVsyncs--;
			}

		} else
		{
			TCCR2B = 0; //stop the tone
 			PORTB &= ~(_BV(SND_PIN));
		}
		
	#if defined(PCB_Version_2) //Added by KJ 22/11/20
		if (bool_scroll == false){ //Added by KJ 17/11/20
		PORTD &= ~(_BV(FSYNC_VOE_PIN));  //Set the Frame SYNC or Video Out En, pin low. Resets the line counters to zero. Added by KJ 17/11/20
		PORTD |= _BV(FSYNC_VOE_PIN);  //Set the Frame SYNC or Video Out En, pin high. Releases the line counters to count. Added by KJ 17/11/20
		do { //Added by KJ 30/11/20
		//Increment the Line counters by character row (8 lines). Has to be done as quickly as possible thus uses single cycle commands. Added by KJ 17/11/20
		PORT_VID |= _BV(VID_PIN); //Set the Video Enable pin High.  Set up to clock the counters. Added by KJ 17/11/20
		PORT_VID &= ~(_BV(VID_PIN)); //Set the Video Enable pin Low. Clock the counters by one pulse.  Added by KJ 17/11/20		
		PORT_VID |= _BV(VID_PIN); //Set the Video Enable pin High.  Set up to clock the counters. Added by KJ 17/11/20
		PORT_VID &= ~(_BV(VID_PIN)); //Set the Video Enable pin Low. Clock the counters by one pulse.  Added by KJ 17/11/20	
		PORT_VID |= _BV(VID_PIN); //Set the Video Enable pin High.  Set up to clock the counters. Added by KJ 17/11/20
		PORT_VID &= ~(_BV(VID_PIN)); //Set the Video Enable pin Low. Clock the counters by one pulse.  Added by KJ 17/11/20		
		PORT_VID |= _BV(VID_PIN); //Set the Video Enable pin High.  Set up to clock the counters. Added by KJ 17/11/20
		PORT_VID &= ~(_BV(VID_PIN)); //Set the Video Enable pin Low. Clock the counters by one pulse.  Added by KJ 17/11/20		
		PORT_VID |= _BV(VID_PIN); //Set the Video Enable pin High.  Set up to clock the counters. Added by KJ 17/11/20
		PORT_VID &= ~(_BV(VID_PIN)); //Set the Video Enable pin Low. Clock the counters by one pulse.  Added by KJ 17/11/20		
		PORT_VID |= _BV(VID_PIN); //Set the Video Enable pin High.  Set up to clock the counters. Added by KJ 17/11/20
		PORT_VID &= ~(_BV(VID_PIN)); //Set the Video Enable pin Low. Clock the counters by one pulse.  Added by KJ 17/11/20	
		PORT_VID |= _BV(VID_PIN); //Set the Video Enable pin High.  Set up to clock the counters. Added by KJ 17/11/20
		PORT_VID &= ~(_BV(VID_PIN)); //Set the Video Enable pin Low. Clock the counters by one pulse.  Added by KJ 17/11/20		
		PORT_VID |= _BV(VID_PIN); //Set the Video Enable pin High.  Set up to clock the counters. Added by KJ 17/11/20
		PORT_VID &= ~(_BV(VID_PIN)); //Set the Video Enable pin Low. Clock the counters by one pulse.  Added by KJ 17/11/20	
		display.scroll_loop--; //Decrement the character row count. Added by KJ 30/11/20
		} while (display.scroll_loop != 0); //Leave when all character rows have been done. Added by KJ 30/11/20
		if (mode==true){  //Added by KJ 30/11/20
			display.scroll_loop=Number_PAL_Scroll_Lines; //When PAL, scroll these number of character rows. Added by KJ 30/11/20
		} //Added by KJ 30/11/20
		else{ //Added by KJ 30/11/20
			display.scroll_loop=Number_NTSC_Scroll_Lines; //When NTSC, scroll these number of character rows. Added by KJ 30/11/20
		} //Added by KJ 30/11/20
		} //Added by KJ 30/11/20
		else{ //Added by KJ 17/11/20
		PORTD &= ~(_BV(FSYNC_VOE_PIN));  //Set the Frame SYNC or Video Out En, pin low. Resets the line counters to zero. Added by KJ 22/11/20
		PORTD |= _BV(FSYNC_VOE_PIN);  //Set the Frame SYNC or Video Out En, pin high. Releases the line counters to count. Added by KJ 22/11/20			
		} //Added by KJ 17/11/20
	#else //Added by KJ 22/11/20	
		PORTD &= ~(_BV(FSYNC_VOE_PIN));  //Set the Frame SYNC or Video Out En pin low. Inhibits the video output. Added by KJ 22/11/20
		//Loads the scrolling value
		if (bool_scroll == false){ //Added by KJ 30/11/20
			if (mode==true){  //Added by KJ 30/11/20	
			//Set the scroll address = 3 for PAL (3 Character rows)
			PORT_PE_A0 |= _BV(PE_A0_PIN); //Set the A0 scroll data = 1. Added by KJ 30/11/20
			PORT_PE_A1 |= _BV(PE_A1_PIN); //Set the A1 scroll data = 1. Added by KJ 30/11/20
			PORT_PE_A2 &= ~(_BV(PE_A2_PIN)); //Set the A2 scroll data = 0. Added by KJ 30/11/20
			PORT_PE_A3 &= ~(_BV(PE_A3_PIN)); //Set the A3 scroll data = 0. Added by KJ 30/11/20
			PORT_PE_A4 &= ~(_BV(PE_A4_PIN)); //Set the A4 scroll data = 0. Added by KJ 30/11/20				
			} //Added by KJ 30/11/20
			else{ //Added by KJ 30/11/20
			//Set the scroll address = 8 for NTSC (8 Character rows)
			PORT_PE_A0 &= ~(_BV(PE_A0_PIN)); //Set the A0 scroll data = 0. Added by KJ 30/11/20
			PORT_PE_A1 &= ~(_BV(PE_A1_PIN)); //Set the A1 scroll data = 0. Added by KJ 30/11/20
			PORT_PE_A2 &= ~(_BV(PE_A2_PIN)); //Set the A2 scroll data = 0. Added by KJ 30/11/20
			PORT_PE_A3 |= _BV(PE_A3_PIN); //Set the A3 scroll data = 1. Added by KJ 30/11/20
			PORT_PE_A4 &= ~(_BV(PE_A4_PIN)); //Set the A4 scroll data = 0. Added by KJ 30/11/20			
			}//Added by KJ 30/11/20
		}//Added by KJ 2/1/21
		else{ //Clear the scroll address for when NO SCROLL is selected. It's possible to instead set a Scroll Character row value here to start the display at a row other than zero.
		PORT_PE_A0 &= ~(_BV(PE_A0_PIN)); //Set the A0 scroll data = 0. Added by KJ 22/11/20
		PORT_PE_A1 &= ~(_BV(PE_A1_PIN)); //Set the A1 scroll data = 0. Added by KJ 22/11/20
		PORT_PE_A2 &= ~(_BV(PE_A2_PIN)); //Set the A2 scroll data = 0. Added by KJ 22/11/20
		PORT_PE_A3 &= ~(_BV(PE_A3_PIN)); //Set the A3 scroll data = 0. Added by KJ 22/11/20
		PORT_PE_A4 &= ~(_BV(PE_A4_PIN)); //Set the A4 scroll data = 0. Added by KJ 22/11/20					
		}
		PORT_PE_ &= ~(_BV(PE_PIN)); //Set the Parallel Load pin low. Set up for the parallel scroll data. Added by KJ 22/11/20
		PORT_VID |= _BV(VID_PIN); //Set the Video Enable pin High.  Set up for clocking the parallel data in. Added by KJ 22/11/20
		PORT_VID &= ~(_BV(VID_PIN)); //Set the Video Enable pin Low. Latch the parallel data in.  Added by KJ 22/11/20		
		PORT_PE_ |= _BV(PE_PIN); //Set the Parallel Load pin high to disable the parallel loading. Added by KJ 22/11/20
		PORTD |= _BV(FSYNC_VOE_PIN);  //Set the Frame SYNC or Video Out En pin high. Releases video output inhibit. Added by KJ 22/11/20			
	#endif //Added by KJ 22/11/20
	}
	else if (display.scanLine == display.vsync_end) {
		OCR1A = _CYCLES_HORZ_SYNC;
		line_handler = &blank_line;
	}
	display.scanLine++;
}


static void inline wait_until(uint8_t time) {
	__asm__ __volatile__ (
			"subi	%[time], 10\n"
			"sub	%[time], %[tcnt1l]\n\t"
		"100:\n\t"
			"subi	%[time], 3\n\t"
			"brcc	100b\n\t"
			"subi	%[time], 0-3\n\t"
			"breq	101f\n\t"
			"dec	%[time]\n\t"
			"breq	102f\n\t"
			"rjmp	102f\n"
		"101:\n\t"
			"nop\n" 
		"102:\n"
		:
		: [time] "a" (time),
		[tcnt1l] "a" (TCNT1L)
	);
}


void render_line6c() {
	#ifndef REMOVE6C
	__asm__ __volatile__ (
		"ADD	r26,r28\n\t"
		"ADC	r27,r29\n\t"
		//save PORTB
		"svprt	%[port]\n\t"
		
		"rjmp	enter6\n"
	"loop6:\n\t"
		"bst	__tmp_reg__,0\n\t"			//8
		"o1bs	%[port]\n"
	"enter6:\n\t"
		"LD		__tmp_reg__,X+\n\t"			//1
		"delay1\n\t"
		"bst	__tmp_reg__,7\n\t"
		"o1bs	%[port]\n\t"
		"delay3\n\t"						//2
		"bst	__tmp_reg__,6\n\t"
		"o1bs	%[port]\n\t"
		"delay3\n\t"						//3
		"bst	__tmp_reg__,5\n\t"
		"o1bs	%[port]\n\t"
		"delay3\n\t"						//4
		"bst	__tmp_reg__,4\n\t"
		"o1bs	%[port]\n\t"
		"delay3\n\t"						//5
		"bst	__tmp_reg__,3\n\t"
		"o1bs	%[port]\n\t"
		"delay3\n\t"						//6
		"bst	__tmp_reg__,2\n\t"
		"o1bs	%[port]\n\t"
		"delay3\n\t"						//7
		"bst	__tmp_reg__,1\n\t"
		"o1bs	%[port]\n\t"
		"dec	%[hres]\n\t"
		"brne	loop6\n\t"					//go too loopsix
		"delay2\n\t"
		"bst	__tmp_reg__,0\n\t"			//8
		"o1bs	%[port]\n"
		
		"svprt	%[port]\n\t"
		BST_HWS
		"o1bs	%[port]\n\t"
		:
		: [port] "i" (_SFR_IO_ADDR(PORT_VID)),
		"x" (display.screen),
		"y" (renderLine),
		[hres] "d" (display.hres)
		: "r16" // try to remove this clobber later...
	);
	#endif
}

void render_line5c() {
	#ifndef REMOVE5C
	__asm__ __volatile__ (
		"ADD	r26,r28\n\t"
		"ADC	r27,r29\n\t"
		//save PORTB
		"svprt	%[port]\n\t"
		
		"rjmp	enter5\n"
	"loop5:\n\t"
		"bst	__tmp_reg__,0\n\t"			//8
		"o1bs	%[port]\n"
	"enter5:\n\t"
		"LD		__tmp_reg__,X+\n\t"			//1
		"bst	__tmp_reg__,7\n\t"
		"o1bs	%[port]\n\t"
		"delay2\n\t"						//2
		"bst	__tmp_reg__,6\n\t"
		"o1bs	%[port]\n\t"
		"delay2\n\t"						//3
		"bst	__tmp_reg__,5\n\t"
		"o1bs	%[port]\n\t"
		"delay2\n\t"						//4
		"bst	__tmp_reg__,4\n\t"
		"o1bs	%[port]\n\t"
		"delay2\n\t"						//5
		"bst	__tmp_reg__,3\n\t"
		"o1bs	%[port]\n\t"
		"delay2\n\t"						//6
		"bst	__tmp_reg__,2\n\t"
		"o1bs	%[port]\n\t"
		"delay1\n\t"						//7
		"dec	%[hres]\n\t"
		"bst	__tmp_reg__,1\n\t"
		"o1bs	%[port]\n\t"
		"brne	loop5\n\t"					//go too loop5
		"delay1\n\t"
		"bst	__tmp_reg__,0\n\t"			//8
		"o1bs	%[port]\n"
		
		"svprt	%[port]\n\t"
		BST_HWS
		"o1bs	%[port]\n\t"
		:
		: [port] "i" (_SFR_IO_ADDR(PORT_VID)),
		"x" (display.screen),
		"y" (renderLine),
		[hres] "d" (display.hres)
		: "r16" // try to remove this clobber later...
	);
	#endif
}

void render_line4c() {
	#ifndef REMOVE4C
	__asm__ __volatile__ (
		"ADD	r26,r28\n\t"
		"ADC	r27,r29\n\t"
		
		"rjmp	enter4\n"
	"loop4:\n\t"
		"lsl	__tmp_reg__\n\t"			//8
		"out	%[port],__tmp_reg__\n\t"
	"enter4:\n\t"
		"LD		__tmp_reg__,X+\n\t"			//1
		"delay1\n\t"
		"out	%[port],__tmp_reg__\n\t"
		"delay2\n\t"						//2
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"
		"delay2\n\t"						//3
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"
		"delay2\n\t"						//4
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"
		"delay2\n\t"						//5
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"
		"delay2\n\t"						//6
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"
		"delay1\n\t"						//7
		"lsl	__tmp_reg__\n\t"
		"dec	%[hres]\n\t"
		"out	%[port],__tmp_reg__\n\t"
		"brne	loop4\n\t"					//go too loop4
		"delay1\n\t"						//8
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"
		"delay3\n\t"
		"cbi	%[port],7\n\t"
		:
		: [port] "i" (_SFR_IO_ADDR(PORT_VID)),
		"x" (display.screen),
		"y" (renderLine),
		[hres] "d" (display.hres)
		: "r16" // try to remove this clobber later...
	);
	#endif
}

// only 16mhz right now!!!
void render_line3c() {
	#ifndef REMOVE3C
	__asm__ __volatile__ (
	".macro byteshift\n\t"
		"LD		__tmp_reg__,X+\n\t"
		"out	%[port],__tmp_reg__\n\t"	//0
		"nop\n\t"
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"	//1
		"nop\n\t"
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"	//2
		"nop\n\t"
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"	//3
		"nop\n\t"
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"	//4
		"nop\n\t"
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"	//5
		"nop\n\t"
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"	//6
		"nop\n\t"
		"lsl	__tmp_reg__\n\t"
		"out	%[port],__tmp_reg__\n\t"	//7
	".endm\n\t"
	
		"ADD	r26,r28\n\t"
		"ADC	r27,r29\n\t"
		
		"cpi	%[hres],30\n\t"		//615
		"breq	skip0\n\t"
		"cpi	%[hres],29\n\t"
		"breq	jumpto1\n\t"
		"cpi	%[hres],28\n\t"
		"breq	jumpto2\n\t"
		"cpi	%[hres],27\n\t"
		"breq	jumpto3\n\t"
		"cpi	%[hres],26\n\t"
		"breq	jumpto4\n\t"
		"cpi	%[hres],25\n\t"
		"breq	jumpto5\n\t"
		"cpi	%[hres],24\n\t"
		"breq	jumpto6\n\t"
	"jumpto1:\n\t"
		"rjmp	skip1\n\t"
	"jumpto2:\n\t"
		"rjmp	skip2\n\t"
	"jumpto3:\n\t"
		"rjmp	skip3\n\t"
	"jumpto4:\n\t"
		"rjmp	skip4\n\t"
	"jumpto5:\n\t"
		"rjmp	skip5\n\t"
	"jumpto6:\n\t"
		"rjmp	skip6\n\t"
	"skip0:\n\t"
		"byteshift\n\t"	//1		\\643
	"skip1:\n\t"
		"byteshift\n\t"	//2
	"skip2:\n\t"
		"byteshift\n\t"	//3
	"skip3:\n\t"
		"byteshift\n\t"	//4
	"skip4:\n\t"
		"byteshift\n\t"	//5
	"skip5:\n\t"
		"byteshift\n\t"	//6
	"skip6:\n\t"
		"byteshift\n\t"	//7
		"byteshift\n\t"	//8
		"byteshift\n\t"	//9
		"byteshift\n\t"	//10
		"byteshift\n\t"	//11
		"byteshift\n\t"	//12
		"byteshift\n\t"	//13
		"byteshift\n\t"	//14
		"byteshift\n\t"	//15
		"byteshift\n\t"	//16
		"byteshift\n\t"	//17
		"byteshift\n\t"	//18
		"byteshift\n\t"	//19
		"byteshift\n\t"	//20
		"byteshift\n\t"	//21
		"byteshift\n\t"	//22
		"byteshift\n\t"	//23
		"byteshift\n\t"	//24
		"byteshift\n\t"	//25
		"byteshift\n\t"	//26
		"byteshift\n\t"	//27
		"byteshift\n\t"	//28
		"byteshift\n\t"	//29
		"byteshift\n\t"	//30
		
		"delay2\n\t"
		"cbi	%[port],7\n\t"
		:
		: [port] "i" (_SFR_IO_ADDR(PORT_VID)),
		"x" (display.screen),
		"y" (renderLine),
		[hres] "d" (display.hres)
		: "r16" // try to remove this clobber later...
	);
	#endif
}
